/*
    Video: https://www.youtube.com/watch?v=oCMOYS71NIU
    Based on Neil Kolban example for IDF: https://github.com/nkolban/esp32-snippets/blob/master/cpp_utils/tests/BLE%20Tests/SampleNotify.cpp
    Ported to Arduino ESP32 by Evandro Copercini

   Create a BLE server that, once we receive a connection, will send periodic notifications.
   The service advertises itself as: 6E400001-B5A3-F393-E0A9-E50E24DCCA9E
   Has a characteristic of: 6E400002-B5A3-F393-E0A9-E50E24DCCA9E - used for receiving data with "WRITE" 
   Has a characteristic of: 6E400003-B5A3-F393-E0A9-E50E24DCCA9E - used to send data with  "NOTIFY"

   The design of creating the BLE server is:
   1. Create a BLE Server
   2. Create a BLE Service
   3. Create a BLE Characteristic on the Service
   4. Create a BLE Descriptor on the characteristic
   5. Start the service.
   6. Start advertising.

   In this example rxValue is the data received (only accessible inside that function).
   And txValue is the data to be sent, in this example just a byte incremented every second. 
*/
#include <BLEDevice.h>
#include <BLEServer.h>
#include <BLEUtils.h>
#include <BLE2902.h>
#include <OneWire.h>
#include <DallasTemperature.h>

BLEServer *pServer = NULL;
BLECharacteristic * pTxCharacteristic;
bool deviceConnected = false;
bool oldDeviceConnected = false;
uint8_t txValue = 0;
String sensorName = "Temperature";

#define sensorPin 5 //sensor pin
#define pin 35 //battery status pin
#define pin_green 23 // green LED
#define pin_red 19 // red LED

float pinValue;
float pinVoltage;

// See the following for generating UUIDs:
// https://www.uuidgenerator.net/

#define SERVICE_UUID           "6e400001-b5a3-f393-e0a9-e50e24dcca9e" // UART service UUID
#define CHARACTERISTIC_UUID_RX "6e400003-b5a3-f393-e0a9-e50e24dcca9e"
#define CHARACTERISTIC_UUID_TX "6e400002-b5a3-f393-e0a9-e50e24dcca9e"


class MyServerCallbacks: public BLEServerCallbacks {
    void onConnect(BLEServer* pServer) {
      deviceConnected = true;
    };

    void onDisconnect(BLEServer* pServer) {
      deviceConnected = false;
    }
};

class MyCallbacks: public BLECharacteristicCallbacks {
    void onWrite(BLECharacteristic *pCharacteristic) {
      std::string rxValue = pCharacteristic->getValue();

      if (rxValue.length() > 0) {
        Serial.println("*********");
        Serial.print("Received Value: ");
        for (int i = 0; i < rxValue.length(); i++)
          Serial.print(rxValue[i]);

        Serial.println();
        Serial.println("*********");
      }
    }
};

// Setup a oneWire instance to communicate with any OneWire device
OneWire oneWire(sensorPin);  
 
// Pass oneWire reference to DallasTemperature library
DallasTemperature sensors(&oneWire);


void setup() {
  sensors.begin();  // Start up the library
  Serial.begin(115200);
  pinMode(pin_green, OUTPUT);
  pinMode(pin_red, OUTPUT);

  // Create the BLE Device
  BLEDevice::init("Arduino ESP32");

  // Create the BLE Server
  pServer = BLEDevice::createServer();
  pServer->setCallbacks(new MyServerCallbacks());

  // Create the BLE Service
  BLEService *pService = pServer->createService(SERVICE_UUID);

  // Create a BLE Characteristic
  pTxCharacteristic = pService->createCharacteristic(
                    CHARACTERISTIC_UUID_TX,
                    BLECharacteristic::PROPERTY_NOTIFY
                  );
                      
  pTxCharacteristic->addDescriptor(new BLE2902());

  BLECharacteristic * pRxCharacteristic = pService->createCharacteristic(
                       CHARACTERISTIC_UUID_RX,
                      BLECharacteristic::PROPERTY_WRITE
                    );

  pRxCharacteristic->setCallbacks(new MyCallbacks());

  // Start the service
  pService->start();

  // Start advertising
  pServer->getAdvertising()->start();
  Serial.println("Starting sensor...");
  delay(3000);
  Serial.println("Warming up...");
  delay(3000); //warming up for 3 seconds
  Serial.println("Waiting a client connection to notify...");
}

void loop() {

  //battery monitor using LEDs
  pinValue = analogRead(pin); //raw digital values
  pinVoltage  = (pinValue/4095)*3.3; //voltage in V

  if (pinVoltage >= 3)
  {
    digitalWrite(pin_green, LOW);   // turn on the green LED
    digitalWrite(pin_red, HIGH);    // turn off the red LED
    Serial.print("voltage : "); Serial.println(pinVoltage);
  }

  if (pinVoltage < 3)
  {
    digitalWrite(pin_red, LOW);   // turn on the red LED
    digitalWrite(pin_green, HIGH);    // turn off the green LED
    Serial.print("voltage : "); Serial.println(pinVoltage);
  }

  // Send the command to get temperatures
  sensors.requestTemperatures(); 

  String str = sensorName + " : " + String(txValue);

    if (deviceConnected) {
        uint8_t arr[30];
        for(int i = 0; i < str.length(); i++)
        {
         arr[i] = str[i];
        }
        Serial.println(str);
        //pTxCharacteristic->setValue(&txValue, 1);
        pTxCharacteristic->setValue(arr, str.length());
        pTxCharacteristic->notify();
        //txValue++;
        txValue = sensors.getTempCByIndex(0);
        delay(2000); // bluetooth stack will go into congestion, if too many packets are sent
  }

    // disconnecting
    if (!deviceConnected && oldDeviceConnected) {
        delay(500); // give the bluetooth stack the chance to get things ready
        pServer->startAdvertising(); // restart advertising
        Serial.println("start advertising");
        oldDeviceConnected = deviceConnected;
    }
    // connecting
    if (deviceConnected && !oldDeviceConnected) {
    // do stuff here on connecting
        oldDeviceConnected = deviceConnected;
    }
}
